{Copyright:      Hagen Reddmann  mailto:HaReddmann@AOL.COM
 Author:         Hagen Reddmann
 Remarks:        freeware, but this Copyright must be included
 known Problems: none
 Version:        3.0
                 Delphi 2-4, designed and testet under D3 and D4
 Description:    Linear Feedback Shift Register (LFSR)
                 Random Number Generator with variable Period
                 from 2^32 -1 to 2^2032 -1, Standard is 2^128 -1
                 with .Seed('', -1) absolutly random
                 The Period have theoretical no effect on the Speed.

 Speed:          ca. 40 Mb/sec of a PII MMX 266 MHz 64Mb RAM

 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


  Speed: all times for PII MMX 266Mhz 64Mb
         theoretical have the Period (Size of LFSR) no effect on the Speed,
         but a greater Period will run faster. (Cache, little Branches on Pentium, etc.)
         except the Period 2^128-1, this use a specially optimized method.

              >   14.5 Mb/sec
              >   40.5 Mb/sec with 128bit LFSR
Version 3.0

  TRandom is now a descend from TProtect, see Unit DECUtil.pas

 }
unit RNG;

interface

//{$I VER.INC}

uses windows,SysUtils, Classes, DECUtil;
const   fmtNONE        =     0;     // allways an Empty String, nothing Action
        fmtDEFAULT     =    -1;     // use DefaultStringFormat
        fmtMIME64      = $1064;     // MIME Base 64
        FStrFMT  : Integer = fmtMIME64; // Default Stringformat
        FStrFMTs : TList = nil;         // registered Stringformats
        IdentityBase        : Word     = $1234;
        fmtCOPY        =     1;     // One to One binary (input = output)
type

  EProtection    = class(Exception);
  EStringFormat  = class(Exception);
  TPAction  = (paEncode, paDecode, paScramble, paCalc, paWipe);
  TPActions = set of TPAction;

 TProtection = class(TObject)
  private
    FRefCount: Integer;
    FProtection: TProtection;
    FActions: TPActions;
    function GetProtection: TProtection;
    procedure SetProtection(Value: TProtection);
  protected
    procedure CodeInit(Action: TPAction); virtual;
    procedure CodeDone(Action: TPAction); virtual;
    procedure CodeBuf(var Buffer; const BufferSize: Integer; Action: TPAction); virtual;
  public
    constructor Create(AProtection: TProtection);
    destructor Destroy; override;

    class function Identity: Word;

    function Release: Integer;
    function AddRef: Integer;

    procedure CodeStream(Source, Dest: TStream; DataSize: Integer; Action: TPAction); virtual;
//    procedure CodeFile(const Source, Dest: String; Action: TPAction); virtual;
//    function  CodeString(const Source: String; Action: TPAction; Format: Integer): String; virtual;
    function  CodeBuffer(var Buffer; BufferSize: Integer; Action: TPAction): Integer; virtual;
    property Protection: TProtection read GetProtection write SetProtection;
    property Actions: TPActions read FActions write FActions default [paEncode..paWipe];
{$IFNDEF VER_D3H}
    property RefCount: Integer read FRefCount;
{$ENDIF}
  end;



  ERandom = class(Exception);
  TRandom = class(TProtection)    // Basicly RNG, equal to Borland's Random()
  private
    FRegister: Integer;
    FPassword: String;
  protected
    FCount: Integer;          // not as private Fields, easier access for descends
    FSize: Integer;
    FBasicSeed: Integer;
    procedure SetSize(Value: Integer); virtual;
    function GetState: String; virtual;
    procedure SetState(Value: String); virtual;
// override TProtect Methods
    procedure CodeInit(Action: TPAction); override;
    procedure CodeDone(Action: TPAction); override;
    procedure CodeBuf(var Buffer; const BufferSize: Integer; Action: TPAction); override;
  public
    constructor Create(const APassword: String; ASize: Integer; ARandomize: Boolean; AProtection: TProtection); virtual;
    destructor Destroy; override;
// set the Seed register
//  Size =  0 -> Seed to initial Value
//  Size <  0 -> Seed to randomness Value, equal to Randomize
//  Size >  0 -> Seed is set to Buffer
    procedure Seed(const ABuffer; ASize: Integer); virtual;
// fill out ABuffer ASize Bytes randomly
    procedure Buffer(var ABuffer; ASize: Integer); virtual;
// gives Random Integer in ARange
    function Int(ARange: Integer): Integer; virtual;
// Stream loading/saving
    procedure SaveToStream(Stream: TStream); virtual;
    procedure LoadFromStream(Stream: TStream); virtual;
// File loading/saving
    procedure SaveToFile(const FileName: String);
    procedure LoadFromFile(const FileName: String);
// Count of Bytes that Int() or Buffer() has generated
    property Count: Integer read FCount write FCount;
// the Size in Bits
    property Size: Integer read FSize write SetSize;
// basicly Seed Value for use in .Seed(), Standard is DefaultSeed
    property BasicSeed: Integer read FBasicSeed write FBasicSeed;
// the internal State as MIMIE Base64 String
    property State: String read GetState write SetState;
  end;
  TStringFormatClass = class of TStringFormat;

  TStringFormat = class(TObject) // for binary one to one convert = fmtCOPY
  public
    class function ToStr(Value: PChar; Len: Integer): String; virtual;
    class function StrTo(Value: PChar; Len: Integer): String; virtual;
    class function Name: String; virtual;
    class function Format: Integer; virtual;
    class function IsValid(Value: PChar; Len: Integer; ToStr: Boolean): Boolean; virtual;
  end;


  TRandom_LFSR = class(TRandom)       // Linear Feedback Shift Register
  private
    FPtr: Integer;                    // Current Position in FRegister
    FLast: Integer;                   // Highest Position in FRegister
    FTable: array[0..255] of Word;    // Lookup Table for FRegister
    FRegister: array[0..255] of Byte; // Linear Feedback Shift Register
    FFunc: procedure(Self: Pointer; var Buffer; Size: Integer);
  protected
    procedure SetSize(Value: Integer); override;
    function GetState: String; override;
    procedure SetState(Value: String); override;
  public
    procedure Seed(const ABuffer; ASize: Integer); override;
    procedure Buffer(var ABuffer; ASize: Integer); override;
  end;

{   Follow the used polynomial's for TRandom_LFSR
     size in bytes of register, XORCode, Polynomial, Period

   4, $F5, x^32   + x^7 + x^5 + x^3 + x^2 + x + 1,   2^32   -1
   5, $9C, x^40   + x^5 + x^4 + x^3 + 1,             2^40   -1
   6, $ED, x^48   + x^7 + x^5 + x^4 + x^2 + x + 1,   2^48   -1
   7, $A9, x^56   + x^7 + x^4 + x^2 + 1,             2^56   -1
   8, $D8, x^64   + x^4 + x^3 + x + 1,               2^64   -1
   9, $FA, x^72   + x^6 + x^4 + x^3 + x^2 + x + 1,   2^72   -1
  10, $F5, x^80   + x^7 + x^5 + x^3 + x^2 + x + 1,   2^80   -1
  12, $BB, x^96   + x^7 + x^6 + x^4 + x^3 + x^2 + 1, 2^96   -1
  15, $E7, x^120  + x^7 + x^6 + x^5 + x^2 + x + 1,   2^120  -1
  16, $E1, x^128  + x^7 + x^2 + x + 1,               2^128  -1
  18, $A9, x^144  + x^7 + x^4 + x^2 + 1,             2^144  -1
  19, $B2, x^152  + x^6 + x^3 + x^2 + 1,             2^152  -1
  20, $B4, x^160  + x^5 + x^3 + x^2 + 1,             2^160  -1
  22, $BD, x^176  + x^7 + x^5 + x^4 + x^3 + x^2 + 1, 2^176  -1
  25, $B4, x^200  + x^5 + x^3 + x^2 + 1,             2^200  -1
  27, $D1, x^216  + x^7 + x^3 + x + 1,               2^216  -1
  38, $FC, x^304  + x^5 + x^4 + x^3 + x^2 + x + 1,   2^304  -1
  40, $D8, x^320  + x^4 + x^3 + x + 1,               2^320  -1
  42, $C9, x^336  + x^7 + x^4 + x + 1,               2^336  -1
  44, $BD, x^352  + x^7 + x^5 + x^4 + x^3 + x^2 + 1, 2^352  -1
  50, $B4, x^400  + x^5 + x^3 + x^2 + 1,             2^400  -1
  51, $FA, x^408  + x^6 + x^4 + x^3 + x^2 + x + 1,   2^408  -1
  55, $D8, x^440  + x^4 + x^3 + x + 1,               2^440  -1
  60, $BB, x^480  + x^7 + x^6 + x^4 + x^3 + x^2 + 1, 2^480  -1
  61, $D8, x^488  + x^4 + x^3 + x + 1,               2^488  -1
  63, $FA, x^504  + x^6 + x^4 + x^3 + x^2 + x + 1,   2^504  -1
  67, $95, x^536  + x^7 + x^5 + x^3 + 1,             2^536  -1
  84, $F6, x^672  + x^6 + x^5 + x^3 + x^2 + x + 1,   2^672  -1
  89, $9C, x^712  + x^5 + x^4 + x^3 + 1,             2^712  -1
  91, $B8, x^728  + x^4 + x^3 + x^2 + 1,             2^728  -1
 103, $FC, x^824  + x^5 + x^4 + x^3 + x^2 + x + 1,   2^824  -1
 141, $D1, x^1128 + x^7 + x^3 + x + 1,               2^1128 -1
 154, $F3, x^1232 + x^7 + x^6 + x^3 + x^2 + x + 1,   2^1232 -1
 254, $A3, x^2032 + x^7 + x^6 + x^2 + 1,             2^2032 -1

  follow various Periods
--------------------------------------------------------------------------------
  2^32-1   = 4294967295
  2^64-1   = 18446744073709551615
  2^128-1  = 340282366920938463463374607431768211455
  2^2032-1 = it's one Number
   49311837877366649323600580884811328064642490645928167773636391338386009428204
   17921935608125537553934278674005267623599165972833122328326583112816221076703
   35702985799671951234310153163915857728680359766210694390385082889078409114931
   66867209378778336289339669574030006474132653643098550122997363890264786354861
   31947843882498538312526670313197249581325688984118966381501107686008635362008
   71492771279798342546336760614070411100118371556871830774626226863061725361438
   46476937385117828689155818331492509954024778049592066494651864619855274961300
   9880449926596639031121858756000207590413184793166384097191709192063287295
--------------------------------------------------------------------------------
}

// Your actual Random Class, per default TRandom_LFSR.Create(128, False)
function RND: TRandom;

// internal used for the random initialization of the Seed Initial Value
// change this to produce Application dependent Randomness
const
  DefaultSeed: Integer = 693258280;

implementation

uses DECConst;

var
  FRND: TRandom = nil;

// avaible Periods for the LFSR
  LFSRPeriod: array[0..33, 0..1] of Word =
   ((   32, $F5), (   40, $9C), (   48, $ED), (   56, $A9),
    (   64, $D8), (   72, $FA), (   80, $F5), (   96, $BB),
    (  120, $E7), (  128, $E1), (  144, $A9), (  152, $B2),
    (  160, $B4), (  176, $BD), (  200, $B4), (  216, $D1),
    (  304, $FC), (  320, $D8), (  336, $C9), (  352, $BD),
    (  400, $B4), (  408, $FA), (  440, $D8), (  480, $BB),
    (  488, $D8), (  504, $FA), (  536, $95), (  672, $F6),
    (  712, $9C), (  728, $B8), (  824, $FC), ( 1128, $D1),
    ( 1232, $F3), ( 2032, $A3));

function RND: TRandom;
begin
  if FRND = nil then
  begin
    FRND := TRandom_LFSR.Create('', 0, False, nil);
    FRND.AddRef;
  end;
  Result := FRND;
end;

procedure TRandom.SetSize(Value: Integer);
begin
  FSize := 32; // allways 32
end;
function RndXORBuffer(Seed: Integer; var Buffer; Size: Integer): Integer; assembler;
asm
      AND     EDX,EDX
      JZ      @@2
      AND     ECX,ECX
      JLE     @@2
      PUSH    EBX
@@1:  XOR     AL,[EDX]
      IMUL    EAX,EAX,134775813
      INC     EAX
      MOV     EBX,EAX
      SHR     EBX,24
      MOV     [EDX],BL
      INC     EDX
      DEC     ECX
      JNZ     @@1
      POP     EBX
@@2:
end;
function RndTimeSeed: Integer; assembler;
var
  SysTime: record
             Year: Word;
             Month: Word;
             DayOfWeek: Word;
             Day: Word;
             Hour: Word;
             Minute: Word;
             Second: Word;
             MilliSeconds: Word;
             Reserved: array [0..7] of Byte;
           end;
  Counter: record
             Lo,Hi: Integer;
           end;
asm
      LEA     EAX,SysTime
      PUSH    EAX
      CALL    GetSystemTime
      MOVZX   EAX,Word Ptr SysTime.Hour
      IMUL    EAX,60
      ADD     AX,SysTime.Minute
      IMUL    EAX,60
      MOVZX   EDX,Word Ptr SysTime.Second
      ADD     EAX,EDX
      IMUL    EAX,1000
      MOV     DX,SysTime.MilliSeconds
      ADD     EAX,EDX
      PUSH    EAX
      LEA     EAX,Counter
      PUSH    EAX
      CALL    QueryPerformanceCounter
      POP     EAX
      XOR     EAX,Counter.Lo
      XOR     EAX,Counter.Hi
end;
function CRC16(CRC: Word; Data: Pointer; DataSize: LongWord): Word; assembler;
asm
         AND    EDX,EDX
         JZ     @Exit
         AND    ECX,ECX
         JLE    @Exit
         PUSH   EBX
         PUSH   EDI
         XOR    EBX,EBX
         LEA    EDI,CS:[OFFSET @CRC16]
@Start:  MOV    BL,[EDX]
         XOR    BL,AL
         SHR    AX,8
         XOR    AX,[EDI + EBX * 2]
         INC    EDX
         DEC    ECX
         JNZ    @Start
         POP    EDI
         POP    EBX
@Exit:   RET
         NOP
@CRC16:  DW     00000h, 0C0C1h, 0C181h, 00140h, 0C301h, 003C0h, 00280h, 0C241h
         DW     0C601h, 006C0h, 00780h, 0C741h, 00500h, 0C5C1h, 0C481h, 00440h
         DW     0CC01h, 00CC0h, 00D80h, 0CD41h, 00F00h, 0CFC1h, 0CE81h, 00E40h
         DW     00A00h, 0CAC1h, 0CB81h, 00B40h, 0C901h, 009C0h, 00880h, 0C841h
         DW     0D801h, 018C0h, 01980h, 0D941h, 01B00h, 0DBC1h, 0DA81h, 01A40h
         DW     01E00h, 0DEC1h, 0DF81h, 01F40h, 0DD01h, 01DC0h, 01C80h, 0DC41h
         DW     01400h, 0D4C1h, 0D581h, 01540h, 0D701h, 017C0h, 01680h, 0D641h
         DW     0D201h, 012C0h, 01380h, 0D341h, 01100h, 0D1C1h, 0D081h, 01040h
         DW     0F001h, 030C0h, 03180h, 0F141h, 03300h, 0F3C1h, 0F281h, 03240h
         DW     03600h, 0F6C1h, 0F781h, 03740h, 0F501h, 035C0h, 03480h, 0F441h
         DW     03C00h, 0FCC1h, 0FD81h, 03D40h, 0FF01h, 03FC0h, 03E80h, 0FE41h
         DW     0FA01h, 03AC0h, 03B80h, 0FB41h, 03900h, 0F9C1h, 0F881h, 03840h
         DW     02800h, 0E8C1h, 0E981h, 02940h, 0EB01h, 02BC0h, 02A80h, 0EA41h
         DW     0EE01h, 02EC0h, 02F80h, 0EF41h, 02D00h, 0EDC1h, 0EC81h, 02C40h
         DW     0E401h, 024C0h, 02580h, 0E541h, 02700h, 0E7C1h, 0E681h, 02640h
         DW     02200h, 0E2C1h, 0E381h, 02340h, 0E101h, 021C0h, 02080h, 0E041h
         DW     0A001h, 060C0h, 06180h, 0A141h, 06300h, 0A3C1h, 0A281h, 06240h
         DW     06600h, 0A6C1h, 0A781h, 06740h, 0A501h, 065C0h, 06480h, 0A441h
         DW     06C00h, 0ACC1h, 0AD81h, 06D40h, 0AF01h, 06FC0h, 06E80h, 0AE41h
         DW     0AA01h, 06AC0h, 06B80h, 0AB41h, 06900h, 0A9C1h, 0A881h, 06840h
         DW     07800h, 0B8C1h, 0B981h, 07940h, 0BB01h, 07BC0h, 07A80h, 0BA41h
         DW     0BE01h, 07EC0h, 07F80h, 0BF41h, 07D00h, 0BDC1h, 0BC81h, 07C40h
         DW     0B401h, 074C0h, 07580h, 0B541h, 07700h, 0B7C1h, 0B681h, 07640h
         DW     07200h, 0B2C1h, 0B381h, 07340h, 0B101h, 071C0h, 07080h, 0B041h
         DW     05000h, 090C1h, 09181h, 05140h, 09301h, 053C0h, 05280h, 09241h
         DW     09601h, 056C0h, 05780h, 09741h, 05500h, 095C1h, 09481h, 05440h
         DW     09C01h, 05CC0h, 05D80h, 09D41h, 05F00h, 09FC1h, 09E81h, 05E40h
         DW     05A00h, 09AC1h, 09B81h, 05B40h, 09901h, 059C0h, 05880h, 09841h
         DW     08801h, 048C0h, 04980h, 08941h, 04B00h, 08BC1h, 08A81h, 04A40h
         DW     04E00h, 08EC1h, 08F81h, 04F40h, 08D01h, 04DC0h, 04C80h, 08C41h
         DW     04400h, 084C1h, 08581h, 04540h, 08701h, 047C0h, 04680h, 08641h
         DW     08201h, 042C0h, 04380h, 08341h, 04100h, 081C1h, 08081h, 04040h
end;
function DefaultStringFormat: Integer;
begin
  Result := FStrFMT;
end;

function StringFormat(Format: Integer): TStringFormatClass;
var
  I: Integer;
begin
  if Format = fmtDefault then Format := DefaultStringFormat;
  Result := nil;
  if FStrFmts <> nil then
    for I := 0 to FStrFMTs.Count-1 do
      if TStringFormatClass(FStrFmts[I]).Format = Format then
      begin
        Result := FStrFMTS[I];
        Exit;
      end;
end;

function StrToFormat(Value: PChar; Len, Format: Integer): String;
var
  Fmt: TStringFormatClass;
begin
  Result := '';
  if (Value = nil) or (Format = fmtNONE) then Exit;
  if Len <  0 then Len := StrLen(Value);
  if Len <= 0 then Exit;
  Fmt := StringFormat(Format);
  if Fmt <> nil then
    if Fmt.IsValid(Value, Len, False) then Result := Fmt.StrTo(Value, Len)
      else raise EStringFormat.CreateFMT(sInvalidFormatString, [FMT.Name])
    else raise EStringFormat.CreateFMT(sStringFormatExists, [Format]);
end;

function TRandom.GetState: String;
var
  CRC: Word;
  M: TMemoryStream;
begin
  M := TMemoryStream.Create;
  try
// write a Randomized Word to begin,
// any Encryption produce allways other outputs
    RndXORBuffer(RndTimeSeed, CRC, SizeOf(CRC));
    M.Write(CRC, SizeOf(CRC));
    M.Write(FSize, SizeOf(FSize));
    M.Write(FBasicSeed, SizeOf(FBasicSeed));
    M.Write(FCount, SizeOf(FCount));
    M.Write(FRegister, SizeOf(FRegister));
    CRC := not CRC16($FFFF, M.Memory, M.Size);
    M.Write(CRC, SizeOf(CRC));
    CRC := $0100; // Version 1 without Protection
    if Protection <> nil then
    begin
      CRC := CRC or 1; // with Protection
      M.Position := 0;
      Protection.CodeStream(M, M, M.Size, paEncode);
      M.Position := M.Size;
    end;
    M.Write(CRC, SizeOf(CRC));
    Result := StrToFormat(M.Memory, M.Size, fmtMIME64);
  finally
    M.Free;
  end;
end;
function FormatToStr(Value: PChar; Len, Format: Integer): String;
var
  Fmt: TStringFormatClass;
begin
  Result := '';
  if (Value = nil) or (Format = fmtNONE) then Exit;
  if Len < 0 then Len := StrLen(Value);
  if Len = 0 then Exit;
  Fmt := StringFormat(Format);
  if Fmt <> nil then
    if Fmt.IsValid(Value, Len, True) then Result := Fmt.ToStr(Value, Len)
      else raise EStringFormat.CreateFMT(sInvalidStringFormat, [FMT.Name])
    else raise EStringFormat.CreateFMT(sStringFormatExists, [Format]);
end;

procedure TRandom.SetState(Value: String);
var
  CRC: Word;
  I: Integer;
  M: TMemoryStream;
begin
  M := TMemoryStream.Create;
  try
    Value := FormatToStr(PChar(Value), Length(Value), fmtMIME64);
    M.Write(PChar(Value)^, Length(Value));
    M.Position := M.Size - SizeOf(CRC);
    M.Read(CRC, SizeOf(CRC));
    if CRC and $FF00 <> $0100 then // it's Version $0100 ?
      raise ERandom.Create(sInvalidRandomStream);
    if CRC and 1 <> 0 then
      if Protection <> nil then
      begin
        M.Position := 0;
        Protection.CodeStream(M, M, M.Size - SizeOf(CRC), paDecode);
      end else raise ERandom.Create(sRandomDataProtected);
    M.Position := M.Size - SizeOf(CRC) * 2;
    M.Read(CRC, SizeOf(CRC));
    if CRC <> not CRC16($FFFF, M.Memory, M.Size - SizeOf(CRC) * 2) then
      raise ERandom.Create(sInvalidRandomStream);
    M.Position := SizeOf(CRC); // skip Dummy Random Word
    M.Read(I, SizeOf(FSize));
    SetSize(I);
    M.Read(FCount, SizeOf(FCount));
    M.Read(FBasicSeed, SizeOf(FBasicSeed));
    M.Read(FRegister, SizeOf(FRegister));
  finally
    M.Free;
  end;
end;

constructor TRandom.Create(const APassword: String; ASize: Integer; ARandomize: Boolean; AProtection: TProtection);
begin
  inherited Create(AProtection);
  FBasicSeed := DefaultSeed;
  FSize := -1;
  FPassword := APassword;
  SetSize(ASize);
  if ASize > 0 then
    if not ARandomize then Seed(PChar(FPassword)^, Length(FPassword))
      else Seed('', -1);
end;

destructor TRandom.Destroy;
begin
  Seed('', 0);
  if Self = FRND then FRND := nil;
  inherited Destroy;
end;

procedure TRandom.Seed(const ABuffer; ASize: Integer);
var
  I: Integer;
  R: PByteArray;
begin
  if (ASize > 0) and (@ABuffer <> nil) then
  begin
    FRegister := FBasicSeed;
    FillChar(FRegister, SizeOf(FRegister), 0);
    R := Addr(FRegister);
    for I := 0 to ASize -1 do
      R[I and 3] := R[I and 3] + TByteArray(ABuffer)[I];
  end else
    if ASize < 0 then FRegister := RndTimeSeed + (FCount +1)
      else FRegister := FBasicSeed;
  if Protection <> nil then
    Protection.CodeBuffer(FRegister, SizeOf(FRegister), paScramble);
end;

function TRandom.Int(ARange: Integer): Integer;
begin
  Buffer(Result, SizeOf(Result));
  if (ARange = 0) or (Result = 0) then Exit;
  if (ARange >= 0) and (Result < 0) then Result := -Result else
    if ARange < 0 then ARange := -ARange;
  Result := Result mod (ARange +1);
  Inc(FCount, SizeOf(Result));
end;

procedure TRandom.Buffer(var ABuffer; ASize: Integer);
begin
  if ASize <= 0 then Exit;
  FillChar(ABuffer, ASize, 0);
  FRegister := RndXORBuffer(FRegister, ABuffer, ASize);
  Inc(FCount, ASize);
  if Protection <> nil then
    Protection.CodeBuffer(ABuffer, ASize, paScramble);
end;

function InsertCR(const Value: String; BlockSize: Integer): String;
var
  I: Integer;
  S,D: PChar;
begin
  if (BlockSize <= 0) or (Length(Value) <= BlockSize) then
  begin
    Result := Value;
    Exit;
  end;
  I := Length(Value);
  SetLength(Result, I + I * 2 div BlockSize + 2);
  S := PChar(Value);
  D := PChar(Result);
  repeat
    Move(S^, D^, BlockSize);
    Inc(S, BlockSize);
    Inc(D, BlockSize);
    D^ := #13; Inc(D);
    D^ := #10; Inc(D);
    Dec(I, BlockSize);
  until I < BlockSize;
  Move(S^, D^, I);
  Inc(D, I);
  SetLength(Result, D - PChar(Result));
end;

procedure TRandom.SaveToStream(Stream: TStream);
var
  I: Integer;
  S,C: String;
begin
  C := ClassName;
  if C[1] = 'T' then Delete(C, 1, 1);
  I := Pos('_', C);
  if I > 0 then Delete(C, 1, I);
  S := InsertCR(State, 64);
  C := C + IntToHex(Length(S), 4) + #13#10 + S;
  Stream.Write(PChar(C)^, Length(C));
end;
function DeleteCR(const Value: String): String;
var
  S,D: PChar;
  I: Integer;
begin
  I := Length(Value);
  SetLength(Result, I);
  D := PChar(Result);
  S := PChar(Value);
  while I > 0 do
  begin
    if (S^ <> #10) and (S^ <> #13) then
    begin
      D^ := S^;
      Inc(D);
    end;
    Inc(S);
    Dec(I);
  end;
  SetLength(Result, D - PChar(Result));
end;

procedure TRandom.LoadFromStream(Stream: TStream);
var
  C,S: String;
  I: Integer;
begin
// write the Name from ClassName (i.E. TRandom_LFSR -> "LFSR"),
// the Size as a 4 Char HEX String and State.
// i.E. LFSR0FCB <CR> State
  C := ClassName;
  if C[1] = 'T' then Delete(C, 1, 1);
  I  := Pos('_', C);
  if I > 0 then Delete(C, 1, I);
  SetLength(S, Length(C));
  Stream.Read(PChar(S)^, Length(C));
  if S <> C then Abort;
  SetLength(S, 6);
  Stream.Read(PChar(S)^, 6);
  SetLength(S, 4);
  I := StrToInt('$' + S);
  SetLength(S, I);
  Stream.Read(PChar(S)^, I);
  State := DeleteCR(S);
end;

procedure TRandom.SaveToFile(const FileName: String);
var
  S: TStream;
begin
  S := TFileStream.Create(FileName, fmCreate);
  try
    SaveToStream(S);
  finally
    S.Free;
  end;
end;

procedure TRandom.LoadFromFile(const FileName: String);
var
  S: TStream;
begin
  S := TFileStream.Create(FileName, fmOpenRead or fmShareDenyNone);
  try
    LoadFromStream(S);
  finally
    S.Free;
  end;
end;

procedure TRandom.CodeInit(Action: TPAction);
begin
  if Action = paWipe then Seed('', -1)
    else Seed(PChar(FPassword)^, Length(FPassword));
  inherited CodeInit(Action);
end;

procedure TRandom.CodeDone(Action: TPAction);
begin
  inherited CodeDone(Action);
  if Action = paWipe then Seed('', -1)
    else Seed(PChar(FPassword)^, Length(FPassword));
end;
procedure XORBuffers(I1, I2: Pointer; Size: Integer; Dest: Pointer); assembler;
asm
       AND   ECX,ECX
       JZ    @@5
       PUSH  ESI
       PUSH  EDI
       MOV   ESI,EAX
       MOV   EDI,Dest
@@1:   TEST  ECX,3
       JNZ   @@3
@@2:   SUB   ECX,4
       JL    @@4
       MOV   EAX,[ESI + ECX]
       XOR   EAX,[EDX + ECX]
       MOV   [EDI + ECX],EAX
       JMP   @@2
@@3:   DEC   ECX
       MOV   AL,[ESI + ECX]
       XOR   AL,[EDX + ECX]
       MOV   [EDI + ECX],AL
       JMP   @@1
@@4:   POP   EDI
       POP   ESI
@@5:
end;

procedure TRandom.CodeBuf(var Buffer; const BufferSize: Integer; Action: TPAction);
const
  maxBufSize = 1024 * 4;
var
  Buf: Pointer;
  BPtr: PByte;
  BSize,CSize: Integer;
begin
  if Action <> paDecode then inherited CodeBuf(Buffer, BufferSize, Action);
  if Action in Actions then
  begin
    BPtr := @Buffer;
    if BPtr = nil then Exit;
    BSize := maxBufSize;
    if BSize > BufferSize then BSize := BufferSize;
    Buf := AllocMem(BSize);
    CSize := BufferSize;
    try
      if Action = paCalc then
      begin
        while CSize > 0 do
        begin
          BSize := CSize;
          if BSize > maxBufSize then BSize := maxBufSize;
          Self.Buffer(Buf^, BSize);
          XORBuffers(Buf, BPtr, BSize, Buf);
          Inc(BPtr, BSize);
          Dec(CSize, BSize);
        end
      end else
      begin
        while CSize > 0 do
        begin
          BSize := CSize;
          if BSize > maxBufSize then BSize := maxBufSize;
          Self.Buffer(Buf^, BSize);
          XORBuffers(Buf, BPtr, BSize, BPtr);
          Inc(BPtr, BSize);
          Dec(CSize, BSize);
        end;
      end;
    finally
      ReallocMem(Buf, 0);
    end;
  end;
  if Action = paDecode then
    inherited CodeBuf(Buffer, BufferSize, Action);
end;

// internal for TRandom_LFSR
procedure LFSRBuf(Self: Pointer; var Buffer; Size: Integer); assembler;
asm
      AND     EDX,EDX    // Buffer = nil ?
      JZ      @@9
      AND     ECX,ECX    // BufferSize <= 0 ?
      JLE     @@9

      PUSH    EDI
      PUSH    ESI
      PUSH    EBX
      PUSH    EBP
      PUSH    EAX

      MOV     EDI,[EAX].TRandom_LFSR.FPtr
      MOV     EBP,[EAX].TRandom_LFSR.FLast
      LEA     ESI,[EAX].TRandom_LFSR.FRegister
      LEA     EBX,[EAX].TRandom_LFSR.FTable
      DEC     EDX

@@1:  MOVZX   EAX,Byte Ptr [ESI + EDI]
      MOV     [EDX + ECX],AL
      MOV     AX,[EBX + EAX * 2]
      MOV     [ESI + EDI],AL
      DEC     EDI
      JS      @@2
      XOR     [ESI + EDI],AH
      ADD     EDI,2
      CMP     EDI,EBP
      JLE     @@3
      XOR     EDI,EDI
      JMP     @@3
@@2:  MOV     EDI,EBP
      XOR     [ESI + EDI],AH
      MOV     EDI,1
@@3:  DEC     ECX
      JNZ     @@1

      POP     EAX
      MOV     [EAX].TRandom_LFSR.FPtr,EDI

      POP     EBP
      POP     EBX
      POP     ESI
      POP     EDI

@@9:
end;

procedure LFSRBuf128(Self: Pointer; var Buffer; Size: Integer); assembler;
asm
      AND     EDX,EDX    // Buffer = nil ?
      JZ      @@9
      AND     ECX,ECX    // BufferSize <= 0 ?
      JLE     @@9

      PUSH    EDI
      PUSH    ESI
      PUSH    EBX
      PUSH    EBP
      PUSH    EAX

      MOV     EDI,[EAX].TRandom_LFSR.FPtr
      LEA     EBP,[EAX].TRandom_LFSR.FTable
      LEA     ESI,[EAX].TRandom_LFSR.FRegister
      DEC     EDX
      XOR     EAX,EAX

@@1:  MOV     AL,[ESI + EDI]
      MOV     BX,[EBP + EAX * 2]
      MOV     [EDX + ECX],AL
      MOV     [ESI + EDI],BL
      DEC     EDI
      AND     EDI,0Fh
      XOR     [ESI + EDI],BH
      ADD     EDI,2
      AND     EDI,0Fh
      DEC     ECX
      JNZ     @@1

      POP     EAX
      MOV     [EAX].TRandom_LFSR.FPtr,EDI

      POP     EBP
      POP     EBX
      POP     ESI
      POP     EDI

@@9:
end;

procedure TRandom_LFSR.SetSize(Value: Integer);

  procedure CalcLFSRTable(XORCode: Byte);
  var
    I,J,Z: Integer;
  begin
    asm // Reverse the bitorder
      XOR   AX,AX
      MOV   AL,XORCode
      MOV   CL,8
@@1:  RCR   AL,1
      RCL   AH,1
      DEC   CL
      JNZ   @@1
      MOV   XORCode,AH
    end;
    FillChar(FTable, SizeOf(FTable), 0);
    for I := 0 to 255 do
    begin
      Z := I;
      for J := 0 to 7 do
      begin
        FTable[I] := FTable[I] shl 1;
        if Z and $80 <> 0 then FTable[I] := FTable[I] xor XORCode;
        Z := Z shl 1;
      end;
    end;
  end;

  procedure DoSet(Index: Integer);
  begin
    FSize := LFSRPeriod[Index, 0];
    FLast := LFSRPeriod[Index, 0] div 8 -1;
    if FSize = 128 then FFunc := LFSRBuf128 else FFunc := LFSRBuf;
    CalcLFSRTable(LFSRPeriod[Index, 1]);
    Seed('', 0);
  end;

var
  I: Integer;
begin
  if Value <= 0 then Value := 128;
  if Value <> FSize then
  begin
    for I := 33 downto 0 do
      if Value >= LFSRPeriod[I, 0] then
      begin
        DoSet(I);
        Exit;
      end;
    DoSet(9);  // The Standard fast 2^128-1 Period
  end;
end;

function TRandom_LFSR.GetState: String;
var
  CRC: Word;
  M: TMemoryStream;
begin
  M := TMemoryStream.Create;
  try
// write randomized Dummy Word
    RndXORBuffer(RndTimeSeed, CRC, SizeOf(CRC));
    M.Write(CRC, SizeOf(CRC));
    M.Write(FSize, SizeOf(FSize));
    M.Write(FRegister, SizeOf(FRegister));
    M.Write(FBasicSeed, SizeOf(FBasicSeed));
    M.Write(FCount, SizeOf(FCount));
    M.Write(FPtr, SizeOf(FPtr));
    M.Write(FLast, SizeOf(FLast));
    CRC := not CRC16($FFFF, M.Memory, M.Size);
    M.Write(CRC, SizeOf(CRC));
    CRC := $0100; // Version 1 without Protection
    if Protection <> nil then
    begin
      CRC := CRC or 1; // with Protection
      M.Position := 0;
      Protection.CodeStream(M, M, M.Size, paEncode);
      M.Position := M.Size;
    end;
    M.Write(CRC, SizeOf(CRC));
    Result := StrToFormat(M.Memory, M.Size, fmtMIME64);
  finally
    M.Free;
  end;
end;

procedure TRandom_LFSR.SetState(Value: String);
var
  P: Integer;
  CRC: Word;
  M: TMemoryStream;
begin
  M := TMemoryStream.Create;
  try
    Value := FormatToStr(PChar(Value), Length(Value), fmtMIME64);
    M.Write(PChar(Value)^, Length(Value));
    M.Position := M.Size - SizeOf(CRC);
    M.Read(CRC, SizeOf(CRC));
    if CRC and $FF00 <> $0100 then // it's Version $0100 ?
      raise ERandom.Create(sInvalidRandomStream);
    if CRC and 1 <> 0 then
      if Protection <> nil then
      begin
        M.Position := 0;
        Protection.CodeStream(M, M, M.Size - SizeOf(CRC), paDecode);
      end else raise ERandom.Create(sRandomDataProtected);
    M.Position := M.Size - SizeOf(CRC) * 2;
    M.Read(CRC, SizeOf(CRC));
    if CRC <> not CRC16($FFFF, M.Memory, M.Size - SizeOf(CRC) * 2) then
      raise ERandom.Create(sInvalidRandomStream);
    M.Position := SizeOf(CRC); // skip Dummy word
    M.Read(P, SizeOf(FSize));
    SetSize(P);
    M.Read(FRegister, SizeOf(FRegister));
    M.Read(FBasicSeed, SizeOf(FBasicSeed));
    M.Read(FCount, SizeOf(FCount));
    M.Read(FPtr, SizeOf(FPtr));
    M.Read(FLast, SizeOf(FLast));
  finally
    M.Free;
  end;
end;

procedure TRandom_LFSR.Seed(const ABuffer; ASize: Integer);
var
  I,S: Integer;
begin
  FPtr := 0;
  if (ASize > 0) and (@ABuffer <> nil) then
  begin
    FillChar(FRegister, SizeOf(FRegister), 0);
    S := FSize div 8;
    for I := 0 to ASize -1 do
      FRegister[I mod S] := FRegister[I mod S] + TByteArray(ABuffer)[I];
  end else
    if ASize < 0 then RndXORBuffer(RndTimeSeed + (FCount +1), FRegister, SizeOf(FRegister))
      else FillChar(FRegister, SizeOf(FRegister), 0);
  RndXORBuffer(FBasicSeed, FRegister, SizeOf(FRegister));
  if Protection <> nil then
    Protection.CodeBuffer(FRegister, SizeOf(FRegister), paScramble);
end;

procedure TRandom_LFSR.Buffer(var ABuffer; ASize: Integer);
begin
  if ASize <= 0 then Exit;
  FFunc(Self, ABuffer, ASize);
  if Protection <> nil then
    Protection.CodeBuffer(ABuffer, ASize, paScramble);
  Inc(FCount, ASize);
end;
function IsObject(AObject: Pointer; AClass: TClass): Boolean;
var
  E: Pointer;
begin
  Result := False;
  if AObject = nil then Exit;
  E := ExceptionClass;
  ExceptionClass := nil;
  try
    if TObject(AObject) is AClass then Result := True;
  except
  end;
  ExceptionClass := E;
end;

function TProtection.GetProtection: TProtection;
begin
  if (FProtection <> nil) and not IsObject(FProtection, TProtection) then FProtection := nil;
  Result := FProtection;
end;

procedure TProtection.SetProtection(Value: TProtection);

  function CheckProtection(P: TProtection): Boolean;
  begin
    Result := True;
    if IsObject(P, TProtection) then
      if P = Self then Result := False
        else Result := CheckProtection(P.FProtection)
  end;

begin
  if Value <> FProtection then
    if CheckProtection(Value) then
    begin
      FProtection.Release;
      FProtection := Value;
      FProtection.AddRef;
    end else raise EProtection.Create(sProtectionCircular)
end;

procedure TProtection.CodeInit(Action: TPAction);
begin
  if Protection <> nil then Protection.CodeInit(Action);
end;

procedure TProtection.CodeDone(Action: TPAction);
begin
  if Protection <> nil then Protection.CodeDone(Action);
end;

procedure TProtection.CodeBuf(var Buffer; const BufferSize: Integer; Action: TPAction);
begin
  if Protection <> nil then Protection.CodeBuf(Buffer, BufferSize, Action);
end;

function TProtection.Release: Integer;
begin
  if IsObject(Self, TProtection) then
  begin
{$IFDEF VER_D3H}
    Result := IUnknown(Self)._Release;
{$ELSE}
    Dec(FRefCount);
    Result := FRefCount;
    if FRefCount = 0 then Destroy;
{$ENDIF}
  end else Result := 0;
end;

function TProtection.AddRef: Integer;
begin
  if IsObject(Self, TProtection) then
  begin
{$IFDEF VER_D3H}
    Result := IUnknown(Self)._AddRef;
{$ELSE}
    Inc(FRefCount);
    Result := FRefCount;
{$ENDIF}
  end else Result := 0;
end;

procedure TProtection.CodeStream(Source, Dest: TStream; DataSize: Integer; Action: TPAction);
const
  maxBufSize = 1024 * 4;
var
  Buf: PChar;
  SPos: Integer;
  DPos: Integer;
  Len: Integer;
//  Size: Integer;
begin
  if Source = nil then Exit;
  if Dest = nil then Dest := Source;
  if DataSize < 0 then
  begin
    DataSize := Source.Size;
    Source.Position := 0;
  end;
  CodeInit(Action);
  Buf := nil;
//  Size := DataSize;
//  DoProgress(Self, 0, Size);
  try
    Buf    := AllocMem(maxBufSize);
    DPos   := Dest.Position;
    SPos   := Source.Position;
    if Action = paCalc then
    begin
      while DataSize > 0 do
      begin
        Len := DataSize;
        if Len > maxBufSize then Len := maxBufSize;
        Len := Source.Read(Buf^, Len);
        if Len <= 0 then Break;
        CodeBuf(Buf^, Len, paCalc);
        Dec(DataSize, Len);
//        DoProgress(Self, Size - DataSize, Size);
      end;
    end else
    begin
      while DataSize > 0 do
      begin
        Source.Position := SPos;
        Len := DataSize;
        if Len > maxBufSize then Len := maxBufSize;
        Len := Source.Read(Buf^, Len);
        SPos := Source.Position;
        if Len <= 0 then Break;
        CodeBuf(Buf^, Len, Action);
        Dest.Position := DPos;
        Dest.Write(Buf^, Len);
        DPos := Dest.Position;
        Dec(DataSize, Len);
//        DoProgress(Self, Size - DataSize, Size);
      end;
    end;
  finally
//    DoProgress(Self, 0, 0);
    ReallocMem(Buf, 0);
    CodeDone(Action);
  end;
end;
constructor TProtection.Create(AProtection: TProtection);
begin
  inherited Create;
  Protection := AProtection;
  FActions := [paEncode..paWipe];
end;

destructor TProtection.Destroy;
begin
  Protection := nil;
  inherited Destroy;
end;

class function TProtection.Identity: Word;
var
  S: String;
begin
  S := ClassName;
  Result := not CRC16(IdentityBase, PChar(S), Length(S));
end;
function TProtection.CodeBuffer(var Buffer; BufferSize: Integer; Action: TPAction): Integer;
begin
  Result := BufferSize;
  CodeInit(Action);
  try
    CodeBuf(Buffer, BufferSize, Action);
  finally
    CodeDone(Action);
  end;
end;
class function TStringFormat.ToStr(Value: PChar; Len: Integer): String;
begin
  SetLength(Result, Len);
  Move(Value^, PChar(Result)^, Len);
end;
class function TStringFormat.StrTo(Value: PChar; Len: Integer): String;
begin
  SetLength(Result, Len);
  Move(Value^, PChar(Result)^, Len);
end;
function GetShortClassName(Value: TClass): String;
var
  I: Integer;
begin
  Result := '';
  if Value = nil then Exit;
  Result := Value.ClassName;
  I := Pos('_', Result);
  if I > 0 then Delete(Result, 1, I);
end;


class function TStringFormat.Name: String;
begin
  if Self = TStringFormat then Result := sFMT_COPY
    else Result := GetShortClassName(Self);
end;

class function TStringFormat.Format: Integer;
begin
  Result := fmtCOPY;
end;

class function TStringFormat.IsValid(Value: PChar; Len: Integer; ToStr: Boolean): Boolean;
begin
  Result := True;
end;



initialization
finalization
  FRND.Release;
end.
